// Copyright 2018 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef V8_WASM_WASM_FEATURES_H_
#define V8_WASM_WASM_FEATURES_H_

// The feature flags are declared in their own header.
#include "src/base/macros.h"
#include "src/wasm/wasm-feature-flags.h"

// All features, including features that do not have flags.
#define FOREACH_WASM_FEATURE FOREACH_WASM_FEATURE_FLAG

namespace v8 {
namespace internal {
class Isolate;
namespace wasm {

// This is an empty type to indicate the end of the {WasmFeatures} struct. We
// use the {end_t} type there to avoid trailing commas that get generated by
// the macro generators. We considered the following alternatives:
// * Add "separators" to the {FOREACH_WASM_FEATURE_FLAGS} between entries. This
//   does not work when we want to have different kinds of flags, e.g. for
//   experimental, staging, and shipped features.
// * Use initialization lists, e.g. construct {WasmFeatures} with
//   "WasmFeatures{true, true, ..., true,}". This solves the comma problem,
//   because trailing commas are allowed here. However, we cannot
//   default-initialize the fields of {WasmFeatures} anymore. This seems
//   error-prone, because default-constructed {WasmFeatures} structs are already
//   used in the code base.
// * Avoid the use of {constexpr}. With that we would be more flexible with how
//   we generate {kAllWasmFeatures} and {kNoWasmFeatures}. These values may be
//   used in performance-critical code, however, e.g. in the decoder or in the
//   interpreter.
struct end_t {};

// Enabled or detected features.
struct WasmFeatures {
#define DECL_FIELD(feat, desc, val) bool feat = false;
  FOREACH_WASM_FEATURE(DECL_FIELD)
#undef DECL_FIELD
  // Marker for the end of the list, see the comment at {end_t}.
  end_t end_;

#define DECL_PARAM(feat, desc, val) bool p##feat,
#define DO_INIT(feat, desc, val) feat(p##feat),
  explicit constexpr WasmFeatures(FOREACH_WASM_FEATURE(DECL_PARAM) end_t)
      : FOREACH_WASM_FEATURE(DO_INIT) end_() {}
#undef DECL_PARAM
#undef DO_INIT
  constexpr WasmFeatures() = default;
};

#define JUST_TRUE(feat, desc, val) true,
static constexpr WasmFeatures kAllWasmFeatures(
    FOREACH_WASM_FEATURE(JUST_TRUE){});
#undef JUST_TRUE

#define JUST_FALSE(feat, desc, val) false,
static constexpr WasmFeatures kNoWasmFeatures(
    FOREACH_WASM_FEATURE(JUST_FALSE){});
#undef JUST_FALSE

static constexpr WasmFeatures kAsmjsWasmFeatures = kNoWasmFeatures;

V8_EXPORT_PRIVATE WasmFeatures WasmFeaturesFromFlags();

// Enables features based on both commandline flags and the isolate.
// Precondition: A valid context must be set in {isolate->context()}.
V8_EXPORT_PRIVATE WasmFeatures WasmFeaturesFromIsolate(Isolate* isolate);

V8_EXPORT_PRIVATE void UnionFeaturesInto(WasmFeatures* dst,
                                         const WasmFeatures& src);

}  // namespace wasm
}  // namespace internal
}  // namespace v8

#endif  // V8_WASM_WASM_FEATURES_H_
